 #include <WiFi.h>
#include <WebServer.h>
#include <ESPmDNS.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET    -1
#define SCREEN_ADDRESS 0x3C

// === Pins per your schematic ===
#define SDA_PIN D4
#define SCL_PIN D5
#define LED_PIN D1  // note: may flicker briefly at boot (UART TX)

// === Your Wi-Fi credentials ===
const char* WIFI_SSID = "your wifi name";
const char* WIFI_PASS = "your wifi password";
const char* MDNS_NAME = "text-display";  // access at http://text-display.local/

Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);
WebServer server(80);

// Simple URL decoder (handles %xx and + for space)
String urlDecode(const String& src) {
  String out;
  out.reserve(src.length());
  for (size_t i = 0; i < src.length(); ++i) {
    char c = src[i];
    if (c == '+') {
      out += ' ';
    } else if (c == '%' && i + 2 < src.length()) {
      char h1 = src[i+1], h2 = src[i+2];
      auto hexVal = [](char ch)->int {
        if (ch >= '0' && ch <= '9') return ch - '0';
        if (ch >= 'A' && ch <= 'F') return ch - 'A' + 10;
        if (ch >= 'a' && ch <= 'f') return ch - 'a' + 10;
        return -1;
      };
      int v1 = hexVal(h1), v2 = hexVal(h2);
      if (v1 >= 0 && v2 >= 0) {
        out += char((v1 << 4) | v2);
        i += 2;
      } else {
        out += c;
      }
    } else {
      out += c;
    }
  }
  return out;
}

void blinkLED(unsigned times = 1, unsigned on_ms = 150, unsigned off_ms = 100) {
  for (unsigned i = 0; i < times; ++i) {
    digitalWrite(LED_PIN, HIGH);
    delay(on_ms);
    digitalWrite(LED_PIN, LOW);
    delay(off_ms);
  }
}

void drawText(const String& msg) {
  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(0,0);

  // very simple wrap to the 128px width (approx 21–22 chars per line at size=1)
  const uint8_t maxChars = 21;
  uint16_t col = 0, row = 0;
  for (size_t i = 0; i < msg.length(); ++i) {
    char ch = msg[i];
    if (ch == '\n' || col >= maxChars) {
      row++;
      col = 0;
      if (ch == '\n') continue; // don't print the newline char
    }
    if (row >= 8) break; // 8 rows of text at size 1 on 64px height
    display.setCursor(col * 6, row * 8); // 6x8 font
    display.write(ch);
    col++;
  }
  display.display();
}

const char HTML_PAGE[] PROGMEM = R"HTML(
<!doctype html>
<html lang="en"><head>
<meta charset="utf-8"/>
<meta name="viewport" content="width=device-width,initial-scale=1"/>
<title>Text Display</title>
<style>
  body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,sans-serif;margin:24px}
  form{display:grid;gap:12px;max-width:520px}
  textarea{width:100%;height:120px;font:14px/1.4 monospace}
  button{padding:10px 14px;font-weight:600;cursor:pointer}
  .row{display:flex;gap:8px;align-items:center}
  .mono{font-family:monospace}
</style>
</head><body>
  <h2>Send Text to OLED</h2>
  <form id="f">
    <textarea name="text" placeholder="Type your message..."></textarea>
    <div class="row">
      <button type="submit">Send</button>
      <button type="button" id="blink">Blink LED</button>
      <span id="status" class="mono"></span>
    </div>
  </form>
<script>
const f = document.getElementById('f');
const st = document.getElementById('status');
f.addEventListener('submit', async (e) => {
  e.preventDefault();
  const text = new FormData(f).get('text') || '';
  st.textContent = 'Sending...';
  const r = await fetch('/msg?text=' + encodeURIComponent(text));
  st.textContent = r.ok ? 'OK' : ('Error ' + r.status);
});
document.getElementById('blink').onclick = async () => {
  st.textContent = 'Blink...';
  const r = await fetch('/blink');
  st.textContent = r.ok ? 'OK' : ('Error ' + r.status);
};
</script>
</body></html>
)HTML";

void handleRoot() {
  server.send(200, "text/html", HTML_PAGE);
}

void handleMsg() {
  if (!server.hasArg("text")) {
    server.send(400, "text/plain", "Missing ?text=");
    return;
  }
  String raw = server.arg("text");
  String decoded = urlDecode(raw);
  Serial.printf("Message received (%u bytes): %s\n", (unsigned)decoded.length(), decoded.c_str());
  blinkLED(2, 120, 80);
  drawText(decoded);
  server.send(200, "text/plain", "OK");
}

void handleBlink() {
  blinkLED(3, 100, 80);
  server.send(200, "text/plain", "Blinked");
}

void setup() {
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LOW);

  Serial.begin(115200);
  delay(200);

  // I2C + OLED
  Wire.begin(SDA_PIN, SCL_PIN);
  if (!display.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS)) {
    Serial.println("SSD1306 init failed!");
    while (true) { delay(1000); }
  }
  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(0, 0);
  display.println("Connecting Wi-Fi...");
  display.display();

  // Wi-Fi
  WiFi.mode(WIFI_STA);
  WiFi.setHostname(MDNS_NAME);
  WiFi.begin(WIFI_SSID, WIFI_PASS);
  unsigned long t0 = millis();
  while (WiFi.status() != WL_CONNECTED) {
    delay(250);
    digitalWrite(LED_PIN, !digitalRead(LED_PIN));
    if (millis() - t0 > 15000) break; // 15s timeout
  }
  digitalWrite(LED_PIN, LOW);

  display.clearDisplay();
  display.setCursor(0, 0);
  if (WiFi.status() == WL_CONNECTED) {
    display.println("Wi-Fi connected");
    display.print("IP: "); display.println(WiFi.localIP());
    Serial.print("IP: "); Serial.println(WiFi.localIP());
  } else {
    display.println("Wi-Fi failed");
    Serial.println("Wi-Fi connect failed.");
  }
  display.display();

  // mDNS (optional)
  if (MDNS.begin(MDNS_NAME)) {
    Serial.println("mDNS started at http://" + String(MDNS_NAME) + ".local/");
  }

  // HTTP routes
  server.on("/", handleRoot);
  server.on("/msg", handleMsg);
  server.on("/blink", handleBlink);
  server.begin();
  Serial.println("HTTP server started");
}

void loop() {
  server.handleClient();
}
